unit IWCompCalendar;

interface

uses
{$IFDEF VSNET}
System.ComponentModel, System.Drawing,
{$ELSE}
{$IFDEF Linux}QControls, {$ELSE}Controls, {$ENDIF}
{$ENDIF}
{$IFDEF Linux}QGraphics, {$ELSE}Graphics, {$ENDIF}
  Classes, SysUtils,
  IWBaseControl, IWControl,
  IWHTMLTag, IWGrids, IWFileReference, IWExtCtrls,
  IWBaseRenderContext, IWScriptEvents, IWColor, IWFont, IWRenderContext, IWBaseInterfaces;

type
  TIWCalendarDateChange = procedure(ADate: TDateTime) of object;
  TIWCalendarGetDateInformation = procedure(ADate: TDateTime; VInformation: TStrings) of object;
  TIWCalendarWeekStarts = (wsMonday, wsTuesday, wsWednesday, wsThursday, wsFriday, wsSaturday, wsSunday);
  TIWCalendarGetDayNames = procedure(var Sunday, Monday, Tuesday, Wednesday, Thursday, Friday, Saturday : string) of object;
  TIWCalendarGetMonthName = procedure(MonthNumber : integer; var MonthName : string) of object;


  TIWCalendarCell = class(TIWGridCell)
  private
    FImageFile: TIWImageFile;
    FCellDate: TDateTime;
    FDateInformation: TStringList;
    procedure SetCellDate(const Value: TDateTime);
    procedure SetDateInformation(const Value: TStringList);
    function RenderContents(AContext: TIWComponent40Context): string;
  public
    constructor Create(ACollection: TCollection); override;
    destructor Destroy; override;
    function RenderSelf(AGrid: TIWCustomGrid; const ARow: Integer;
      const AColumn: Integer; AContext: TIWComponent40Context; AText: string = ''): TIWHTMLTag; override;

    property ImageFile: TIWImageFile read FImageFile;

    property CellDate: TDateTime read FCellDate write SetCellDate;

    property DateInformation: TStringList read FDateInformation write SetDateInformation;
  end;

  TIWCalendarCells = class(TOwnedCollection)
  public
    function Add: TIWCalendarCell;
    function Owner: TPersistent;
  end;

{$IFDEF VSNET}
{$R icons\Atozed.Intraweb.TIWCalendar.bmp}
  TIWCalendar = class;
  [ToolboxItem(true), ToolboxBitmap(typeof(TIWCalendar), 'TIWCalendar.bmp')]
{$ENDIF}
  TIWCalendar = class(TIWCustomGrid, IIWSubmitControl)
  private
    { Private declarations }
    FSubmitParam : String;
    FCurrentMonth: word;
    FCurrentYear: word;
    FColumnCount: integer;
    FRowCount: integer;
    FCellCollection: TIWCalendarCells;
    FCurrentDayImage: TIWFileReference;
    FNextMonthImage: TIWFileReference;
    FPreviousMonthImage: TIWFileReference;

    FCalendarColor: TIWColor;
    FCalendarHeaderColor: TIWColor;
    FCalendarFont: TIWFont;
    FCalendarHeaderFont: TIWFont;

    FOnDateChange: TIWCalendarDateChange;
    FOnGetDateInformation: TIWCalendarGetDateInformation;
    FOnGetDayNames : TIWCalendarGetDayNames;
    FOnGetMonthName : TIWCalendarGetMonthName;
    FAlternateCalendarColor: TIWColor;
    FCheckerBoard: boolean;
    FSelectedDate: TDateTime;
    FLongDayNames: boolean;
    FDisplayYear : boolean;

    FDayNames : array [1..7] of string;

    FCaptionToday: string;
    FCaptionNext: string;
    FCaptionPrevious: string;

    FStartDate: TDateTime;
    FWeekStarts: TIWCalendarWeekStarts;

    FLocaleID : Integer;

    procedure SetCurrentDayImage(const Value: TIWFileReference);
    procedure SetNextMonthImage(const Value: TIWFileReference);
    procedure SetPreviousMonthImage(const Value: TIWFileReference);
    procedure SetOnDateChange(const Value: TIWCalendarDateChange);
    function GetCurrentDayImage: TIWFileReference;
    function GetNextMonthImage: TIWFileReference;
    function GetPreviousMonthImage: TIWFileReference;
    procedure SetCalendarFont(const Value: TIWFont);
    procedure SetCalendarHeaderFont(const Value: TIWFont);
    procedure SetStartDate(const Value: TDateTime);
  protected
    { Protected declarations }
    procedure DoCellClick(const ARow: Integer; const AColumn: Integer);  virtual;
    procedure DoGetDateInformation(ADate: TDateTime; VInformation: TStrings);   virtual;
    procedure LoadMonth(AContext: TIWBaseComponentContext); virtual;
    procedure GoPreviousMonth; virtual;
    procedure GoNextMonth(AContext: TIWBaseComponentContext); virtual;
    function GetCell(const ARow, AColumn: Integer): TIWCalendarCell;
    function CellExists(const ARow, AColumn: Integer): Boolean;
    procedure Submit(const AValue: string); override;
    function GetDayName(DayNo: integer): string; virtual;
    procedure RenderCells(AContext: TIWComponent40Context; AGridTag: TIWHTMLTag); override;
    procedure DoGetDayNames;virtual;
    procedure DoRenderCell(ACell: TIWGridCell; const ARow: integer; const AColumn: integer);

    function IsValidCell(ACell: TIWGridCell): Boolean; override;

    procedure DefineProperties(Filer: TFiler); override;
    procedure LoadCaptionToday(Reader: TReader);
    procedure StoreCaptionToday(Writer: TWriter);
    procedure LoadCaptionPrevious(Reader: TReader);
    procedure StoreCaptionPrevious(Writer: TWriter);
    procedure LoadCaptionNext(Reader: TReader);
    procedure StoreCaptionNext(Writer: TWriter);

    procedure InitControl; override;
    procedure InitDesignTime; override;

  {$IFDEF CLR}
  strict protected
  {$ELSE}
  protected
  {$ENDIF}
    procedure Dispose(ADispose: Boolean); override;
  public
     { Public declarations }

    function RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag; override;
    function GetSubmitParam : String;

    procedure SetHeaderCells;
    property Cell[const ARow: Integer; const AColumn: Integer]: TIWCalendarCell read GetCell;

    property LocaleID : Integer read FLocaleID write FLocaleID;
  published
     { Published declarations }
    {$IFDEF VSNET}
    [DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]
    {$ENDIF}
    property CurrentDayImage: TIWFileReference read GetCurrentDayImage write SetCurrentDayImage;
    {$IFDEF VSNET}
    [DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]
    {$ENDIF}
    property PreviousMonthImage: TIWFileReference read GetPreviousMonthImage write SetPreviousMonthImage;
    {$IFDEF VSNET}
    [DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]
    {$ENDIF}
    property NextMonthImage: TIWFileReference read GetNextMonthImage write SetNextMonthImage;
    property CalendarColor: TIWColor read FCalendarColor write FCalendarColor{$IFNDEF VSNET}default clWebGAINSBORO{$ENDIF};
    property AlternateCalendarColor: TIWColor read FAlternateCalendarColor write FAlternateCalendarColor{$IFNDEF VSNET}default clWebLAVENDER{$ENDIF};
    property CalendarHeaderColor: TIWColor read FCalendarHeaderColor write FCalendarHeaderColor{$IFNDEF VSNET}default clWebLIGHTSTEELBLUE{$ENDIF};
    property CalendarFont: TIWFont read FCalendarFont write SetCalendarFont;
    property CalendarHeaderFont: TIWFont read FCalendarHeaderFont write SetCalendarHeaderFont;
    property CheckerBoard: boolean read FCheckerBoard write FCheckerBoard default False;
    property SelectedDate: TDateTime read FSelectedDate;
    property FriendlyName;
    property StartDate: TDateTime read FStartDate write SetStartDate;
    property UseLongDayNames: boolean read FLongDayNames write FLongDayNames default False;
    property CaptionToday: string read FCaptionToday write FCaptionToday stored False;
    property CaptionPrevious: string read FCaptionPrevious write FCaptionPrevious stored False;
    property CaptionNext: string read FCaptionNext write FCaptionNext stored False;
    property WeekStarts: TIWCalendarWeekStarts read FWeekStarts write FWeekStarts default wsMonday;
    property OnDateChange: TIWCalendarDateChange read FOnDateChange write SetOnDateChange;
    property OnGetDateInformation: TIWCalendarGetDateInformation read FOnGetDateInformation write FOnGetDateInformation;
    property OnGetDayNames: TIWCalendarGetDayNames read FOnGetDayNames write FOnGetDayNames;
    property OnGetMonthName : TIWCalendarGetMonthName read FOnGetMonthName write FOnGetMonthName;
    property DisplayYear : boolean read FDisplayYear write FDisplayYear;
  end;

implementation

uses
  IWCompMemo, IWGridCommon, IWCompListbox, IWForm, IWTypes,
  SWSystem, SWStrings, IWResourceStrings, IWMarkupLanguageTag;

{ TIWCalendar }

function TIWCalendar.CellExists(const ARow, AColumn: Integer): Boolean;
begin
  if (AColumn >= FColumnCount) or (ARow >= FRowCount) or (AColumn < 0) or (ARow < 0) then
  begin
    raise Exception.Create(RSIndexOutOfBounds);
  end;
  Result := FCells[ARow, AColumn] <> nil;
end;

procedure TIWCalendar.InitControl;
var
  f: integer;
begin
  inherited;

  FCellCollection := TIWCalendarCells.Create(Self, TIWCalendarCell);

  FCurrentDayImage := TIWFileReference.Create;
  FNextMonthImage := TIWFileReference.Create;
  FPreviousMonthImage := TIWFileReference.Create;

  FCalendarColor := fromTColor(clWebGAINSBORO);
  FAlternateCalendarColor := fromTColor(clWebLAVENDER);

  FCalendarHeaderColor := fromTColor(clWebLIGHTSTEELBLUE);
  FCalendarFont := IWFont.TIWFont.Create;
  FCalendarHeaderFont := IWFont.TIWFont.Create;

  FCheckerBoard := False;
  FDisplayYear := True;

  FCaptionNext := RSNext;
  FCaptionPrevious := RSPrevious;
  FLongDayNames := False;
  FStartDate := Now;

  FRowCount := 8;
  FColumnCount := 7;
  SetLength(FCells, FRowCount);
  for f := 0 to Pred(FRowCount) do
    SetLength(FCells[f], FColumnCount);

  FLocaleID := SysLocale.DefaultLCID;
end;

procedure TIWCalendar.InitDesignTime;
begin
  inherited;

  SetStartDate(FStartDate);

  if IsDesignMode then
  begin
    FCaptionToday := RSToday;
  end;
end;

procedure TIWCalendar.Dispose(ADispose: Boolean);
begin
  FreeAndNil(FCellCollection);
  FreeAndNil(FCurrentDayImage);
  FreeAndNil(FNextMonthImage);
  FreeAndNil(FPreviousMonthImage);
  FreeAndNil(FCalendarFont);
  FreeAndNil(FCalendarHeaderFont);
  inherited;
end;

procedure TIWCalendar.DoCellClick(const ARow, AColumn: Integer);
var
  LCell: TIWCalendarCell;
begin
  LCell := Cell[ARow, AColumn];

  if (ARow = 0) and (AColumn = 0) then
  begin
    GoPreviousMonth;
  end;

  if (ARow = 0) and (AColumn = 6) then
  begin
    GoNextMonth(nil);
  end;

  FSelectedDate := LCell.CellDate;

  if (LCell.CellDate <> 0) and Assigned(OnDateChange) then
    OnDateChange(LCell.CellDate);
end;

procedure TIWCalendar.GoNextMonth(AContext: TIWBaseComponentContext);
begin
  if FCurrentMonth = 12 then
  begin
    FCurrentMonth := 1;
    Inc(FCurrentYear);
  end
  else
    Inc(FCurrentMonth);
  LoadMonth(AContext);
end;

procedure TIWCalendar.GoPreviousMonth;
begin
  if FCurrentMonth = 1 then
  begin
    FCurrentMonth := 12;
    Dec(FCurrentYear);
  end
  else
    Dec(FCurrentMonth);
  LoadMonth(nil);
end;

function TIWCalendar.GetCell(const ARow, AColumn: Integer): TIWCalendarCell;
begin
  if not CellExists(ARow, AColumn) then
  begin
    FCells[ARow, AColumn] := FCellCollection.Add;
    FCells[ARow, AColumn].Font.Assign(Self.WebFont);
  end;
  Result := TIWCalendarCell(FCells[ARow, AColumn]);
end;

procedure TIWCalendar.LoadMonth(AContext: TIWBaseComponentContext);
var
  f, g, h: integer;
  CurrentDay: word;
  ADate: TDateTime;
  LAlternating: boolean;
  LMonthName : string;
  LYear, LMonth, LDay : word;
{$IFDEF VCL7ORABOVE}
  LFormatSettings : TFormatSettings;
{$ENDIF}
begin
  // Get day names
  DoGetDayNames;

  // Setup calendar header
  for f := 0 to 6 do
    with Cell[1, f] do
    begin
      Text := GetDayName((f + Ord(FWeekStarts)) mod 7);
      Alignment := taCenter;
      Height := '20%';
      VAlign := vaMiddle;
      if not (f in [0, 6]) then
        Width := '14%'
      else
        Width := '15%';
    end;

  Cell[0, 0].Clickable := TRUE;
  Cell[0, 0].Alignment := taLeftJustify;
  Cell[0, 0].VAlign := vaMiddle;

  Cell[0, 6].Clickable := TRUE;
  Cell[0, 6].Alignment := taRightJustify;
  Cell[0, 6].VAlign := vaMiddle;

  if (FPreviousMonthImage <> nil) and
    (FPreviousMonthImage.Location('') <> '') then
    Cell[0, 0].ImageFile.ImageFile := FPreviousMonthImage
  else
    Cell[0, 0].Text := FCaptionPrevious;

  if (FNextMonthImage <> nil) and
    (FNextMonthImage.Location('') <> '') then
    Cell[0, 6].ImageFile.ImageFile := FNextMonthImage
  else
    Cell[0, 6].Text := FCaptionNext;

  // Setup calendar days
  CurrentDay := 1;
  Lalternating := True;

  ADate := EncodeDate(FCurrentYear, FCurrentMonth, CurrentDay);
  for f := Pred(FRowCount) - 5 to Pred(FRowCount) do
    for g := 0 to Pred(FColumnCount) do
      with Cell[f, g] do
      begin
        Alignment := taCenter;
        VAlign := vaMiddle;
        Text := '';
        Height := '10%';

        if FCheckerBoard then
        begin
          if LAlternating then
            BGColor := FCalendarColor
          else
            BGColor := FAlternateCalendarColor;
          LAlternating := not LAlternating;
        end
        else
          BGColor := FCalendarColor;

        Font.Assign(FCalendarFont);
        Clickable := False;
        ImageFile.ImageFile.Filename := '';

        CellDate := 0;
        DateInformation.Clear;

        if CurrentDay <= MonthDays[IsLeapYear(FCurrentYear)][FCurrentMonth] then
        begin
          ADate := EncodeDate(FCurrentYear, FCurrentMonth, CurrentDay);
          if (DayOfWeek(ADate) + 5) mod 7 = (g + Ord(FWeekStarts)) mod 7 then
          begin
            CellDate := ADate;
            DoGetDateInformation(CellDate, DateInformation);

            Text := IntToStr(CurrentDay);
            for h := 0 to Pred(DateInformation.Count) do
            begin
              Text := Format('%s<BR>%s', [Text, DateInformation.Strings[h]]);
            end;

            Clickable := TRUE;
            if ADate = Date then
            begin
              if (FCurrentDayImage <> nil) and
                (FCurrentDayImage.Location('') <> '') then
                ImageFile.ImageFile := FCurrentDayImage
              else
                if Trim(FCaptionToday) <> '' then
                begin
                  Text := Format('%s<BR>%s', [FCaptionToday, Text]);
                end;
            end;
            Inc(CurrentDay);
          end;
        end;
      end;

  // Setup date information cell

{$IFDEF VCL7ORABOVE}
  GetLocaleFormatSettings(FLocaleID,LFormatSettings);
{$ENDIF}

  // Get month name
{$IFDEF VCL7ORABOVE}
  LMonthName := FormatDateTime('mmmm', ADate,LFormatSettings);
{$ELSE}
  LMonthName := FormatDateTime('mmmm', ADate);
{$ENDIF}

  if Assigned(FOnGetMonthName) then
  begin
    DecodeDate(ADate, LYear, LMonth, LDay);
    FOnGetMonthName(LMonth, LMonthName);
  end;

{$IFDEF VCL7ORABOVE}
  Cell[0, 1].Text := iif(FDisplayYear, Format('%s %s', [LMonthName, FormatDateTime('yyyy', ADate,LFormatSettings)]), LMonthName);
{$ELSE}
  Cell[0, 1].Text := iif(FDisplayYear, Format('%s %s', [LMonthName, FormatDateTime('yyyy', ADate)]), LMonthName);
{$ENDIF}


  Cell[0, 1].Alignment := taCenter;
  Cell[0, 1].Font.Assign(FCalendarHeaderFont);

  DoRefreshControl := True;
end;

procedure TIWCalendar.RenderCells(AContext: TIWComponent40Context; AGridTag: TIWHTMLTag);
var
  i: integer;
  j: integer;
  LText: string;
begin
  inherited;

  //Render Header into a separate <TABLE>, that is its content is independent in width from the day columns
  with AGridTag.Contents.AddTag('TR') do
    with contents.AddTag('TD') do
    begin
      //span over all day colums
      AddIntegerParam('COLSPAN', 7);
      with contents.AddTag('Table') do
      begin
        //Header-Table uses full width
        AddStringParam('width', '100%');
        AddIntegerParam('cellpadding', 0);
        AddIntegerParam('cellspacing', 0);
        with Contents.AddTag('TR') do
        begin
          with TIWCalendarCell(FCells[0, 0]) do
          begin
            with SubmitLink(RenderContents(AContext), '0_0', False, AContext.Browser) do
            begin
              LText := Render; // TODO: FIX (don't use strings)
              Free;
            end;
            Contents.AddTagAsObject(RenderSelf(Self, 0, 0, AContext, LText));
          end;

          with TIWCalendarCell(FCells[0, 1]) do
            Contents.AddTagAsObject(RenderSelf(Self, 0, 1, AContext));

          with TIWCalendarCell(FCells[0, 6]) do
          begin
            with SubmitLink(RenderContents(AContext), '0_6', False, AContext.Browser) do
            begin
              LText := Render; // TODO: FIX (don't use strings)
              Free;
            end;
            Contents.AddTagAsObject(RenderSelf(Self, 0, 6, AContext, LText))
          end;
        end;
      end;
    end;

  // Render calendar
  for i := 1 to FRowCount - 1 do
  begin
    with AGridTag.Contents.AddTag('TR') do
    begin
      for j := 0 to FColumnCount - 1 do
      begin
        if CellExists(i, j) then
        begin
          with TIWCalendarCell(FCells[i, j]) do
          begin
            if Visible then
            begin
              if Clickable then
              begin
                with SubmitLink(RenderContents(AContext), IntToStr(i) + '_' + IntToStr(j), False, AContext.Browser) do
                begin
                  LText := Render; // TODO: FIX (don't use strings)
                  Free;
                end;
                Contents.AddTagAsObject(RenderSelf(Self, i, j, AContext, LText));
              end
              else
              begin
                Contents.AddTagAsObject(RenderSelf(Self, i, j, AContext));
              end;
            end;
          end;
        end
        else
        begin
              // Not &nbsp; - only if cell exists but is empty
          with Contents.AddTag('TD') do
            Contents.AddText('');
        end;
      end;
    end;
  end;
end;

function TIWCalendar.GetSubmitParam: String;
begin
  Result := FSubmitParam;
end;

function TIWCalendar.IsValidCell(ACell: TIWGridCell): Boolean;
begin
  result := false;
end;

function TIWCalendar.RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag;
begin
  LoadMonth(AContext);
  SetHeaderCells;

  // If Control property is used.
  Result := TIWHTMLTag.CreateTag('TABLE');
  try
    Result.AddStringParam('ID', HTMLName);
    Result.AddColor('BGColor', BGColor);
    Result.AddIntegerParam('CELLPADDING', CellPadding);
    Result.AddIntegerParam('CELLSPACING', CellSpacing);
    if UseSize then begin
      Result.AddIntegerParam('WIDTH', iif(UseFrame, Width - FrameBuffer, Width));
    end;
    set_RenderSize(UseSize);
    Result.Contents.AddText(iif(Caption, '<CAPTION>' + Caption + '</CAPTION>'));
    Result.Contents.AddText(iif(Summary, '<SUMMARY>' + Caption + '</SUMMARY>'));
    RenderCells(TIWComponent40Context(AContext), Result);
  except
    FreeAndNil(Result);
    raise;
  end;
end;

procedure TIWCalendar.SetCurrentDayImage(const Value: TIWFileReference);
begin
  FCurrentDayImage.Assign(Value);
end;

procedure TIWCalendar.SetNextMonthImage(const Value: TIWFileReference);
begin
  FNextMonthImage.Assign(Value);
end;

procedure TIWCalendar.SetOnDateChange(const Value: TIWCalendarDateChange);
begin
  FOnDateChange := Value;
end;

procedure TIWCalendar.SetPreviousMonthImage(const Value: TIWFileReference);
begin
  FPreviousMonthImage.Assign(Value);
end;

procedure TIWCalendar.Submit(const AValue: string);
var
  s: string;
  LColumn: Integer;
  LRow: Integer;
begin
  FSubmitParam := AValue;
  s := AValue;
  LRow := StrToInt(Fetch(s, '_'));
  LColumn := StrToInt(s);
  DoCellClick(LRow, LColumn);
end;

function TIWCalendar.GetCurrentDayImage: TIWFileReference;
begin
  Result := FCurrentDayImage;
end;

function TIWCalendar.GetNextMonthImage: TIWFileReference;
begin
  Result := FNextMonthImage;
end;

function TIWCalendar.GetPreviousMonthImage: TIWFileReference;
begin
  Result := FPreviousMonthImage;
end;

procedure TIWCalendar.SetCalendarFont(const Value: TIWFont);
begin
  FCalendarFont.Assign(Value);
end;

procedure TIWCalendar.SetCalendarHeaderFont(const Value: TIWFont);
begin
  FCalendarHeaderFont.Assign(Value);
end;

procedure TIWCalendar.SetHeaderCells;
var
  f: integer;
begin
  for f := 0 to Pred(FColumnCount) do
  begin
    Cell[1, f].Font.Assign(FCalendarHeaderFont);
    Cell[1, f].BGColor := FCalendarHeaderColor;
  end;
end;

procedure TIWCalendar.DoGetDateInformation(ADate: TDateTime;
  VInformation: TStrings);
begin
  if Assigned(OnGetDateInformation) then
    OnGetDateInformation(ADate, VInformation);
end;

function TIWCalendar.GetDayName(DayNo: integer): string;
begin
  DayNo := (DayNo + 1) mod 7 + 1;
  Result := FDayNames[DayNo];
end;

procedure TIWCalendar.SetStartDate(const Value: TDateTime);
var
  LDay: Word;
  LTodayYear, LTodayMonth: Word;
begin
  FStartDate := Value;
  DecodeDate(Value, FCurrentYear, FCurrentMonth, LDay);
  DecodeDate(Now, LTodayYear, LTodayMonth, LDay);
  if (FCurrentMonth = LTodayMonth) and (FCurrentYear = LTodayYear) then
    FSelectedDate := EncodeDate(LTodayYear, LTodayMonth, LDay)
  else
    FSelectedDate := EncodeDate(FCurrentYear, FCurrentMonth, 1);

  if not IsDesignMode then
    LoadMonth(nil);
end;

procedure TIWCalendar.DoGetDayNames;
var
  f : integer;
{$IFDEF VCL7ORABOVE}
  LFormatSettings : TFormatSettings;
{$ENDIF}
begin
{$IFDEF VCL7ORABOVE}
  GetLocaleFormatSettings(FLocaleID,LFormatSettings);
{$ENDIF}
  for f := 1 to 7 do
  begin
{$IFDEF VCL7ORABOVE}
    FDayNames[f] := iif(FLongDayNames, LFormatSettings.LongDayNames[f], LFormatSettings.ShortDayNames[f]);
{$ELSE}
    FDayNames[f] := iif(FLongDayNames, LongDayNames[f], ShortDayNames[f]);
{$ENDIF}
  end;

  if Assigned(FOnGetDayNames) then
  begin
    FOnGetDayNames(FDayNames[1], FDayNames[2], FDayNames[3], FDayNames[4], FDayNames[5], FDayNames[6], FDayNames[7]);
  end;
end;

procedure TIWCalendar.DoRenderCell(ACell: TIWGridCell; const ARow: integer; const AColumn: integer);
begin
  if Assigned(OnRenderCell) then
  begin
    OnRenderCell(ACell, ARow, AColumn);
  end;
end;

procedure TIWCalendar.DefineProperties(Filer: TFiler);
begin
 inherited DefineProperties(Filer);
 Filer.DefineProperty('CaptionToday', LoadCaptionToday, StoreCaptionToday, FCaptionToday <> RSToday);
 Filer.DefineProperty('CaptionPrevious', LoadCaptionPrevious, StoreCaptionPrevious, FCaptionPrevious <> RSPrevious);
 Filer.DefineProperty('CaptionNext', LoadCaptionNext, StoreCaptionNext, FCaptionNext <> RSNext);
end;

procedure TIWCalendar.LoadCaptionToday(Reader: TReader);
begin
 FCaptionToday := Reader.ReadString;
end;

procedure TIWCalendar.StoreCaptionToday(Writer: TWriter);
begin
 Writer.WriteString(FCaptionToday);
end;

procedure TIWCalendar.LoadCaptionPrevious(Reader: TReader);
begin
 FCaptionPrevious := Reader.ReadString;
end;

procedure TIWCalendar.StoreCaptionPrevious(Writer: TWriter);
begin
 Writer.WriteString(FCaptionPrevious);
end;

procedure TIWCalendar.LoadCaptionNext(Reader: TReader);
begin
 FCaptionNext := Reader.ReadString;
end;

procedure TIWCalendar.StoreCaptionNext(Writer: TWriter);
begin
 Writer.WriteString(FCaptionNext);
end;

{ TIWCalendarCell }

constructor TIWCalendarCell.Create(ACollection: TCollection);
begin
  inherited Create(ACollection);
  FImageFile := IWExtCtrls.TIWImageFile.Create{$IFNDEF VSNET}(nil){$ENDIF};
  FCellDate := 0;
  FDateInformation := TStringList.Create;
end;

destructor TIWCalendarCell.Destroy;
begin
  FImageFile.Free;
  FDateInformation.Free;
  inherited Destroy;
end;

function TIWCalendarCell.RenderContents(AContext: TIWComponent40Context): string;
var
  ATag: TIWHTMLTag;
begin
  ATag := TIWHTMLTag.Create(TIWMarkupLanguageElement(nil)); try

    if (FImageFile.ImageFile.Location('') <> '') then
    begin
      FImageFile.UseSize := FALSE;
      ATag.Contents.AddTagAsObject(FImageFile.RenderHTML(AContext));
      if Text <> '' then
        ATag.Contents.AddTag('BR');
    end;

    if Text <> '' then
      ATag.Contents.AddText(Text);

    Result := ATag.Render; // TODO: FIX (don't use strings)
  finally
    ATag.Free;
  end;
end;

function TIWCalendarCell.RenderSelf(AGrid: TIWCustomGrid; const ARow,
  AColumn: Integer; AContext: TIWComponent40Context; AText: string): TIWHTMLTag;
var
  LVAlign: string;
begin
  if not Visible then
  begin
    Result := TIWHTMLTag.CreateTag('TD');
    try
      Result.Contents.AddText('&nbsp;');
    except
      FreeAndNil(Result);
      raise;
    end;
  end
  else
  begin
    Grid := AGrid;
    if AGrid is TIWCalendar then
    begin
      TIWCalendar(AGrid).DoRenderCell(Self, ARow, AColumn);
    end;

    try
        // VAlign
      LVAlign := '';
      case FVAlign of
        vaMiddle: LVAlign := 'VAlign=middle';
        vaTop: LVAlign := 'VAlign=top';
        vaBottom: LVAlign := 'VAlign=bottom';
        vaBaseline: LVAlign := 'VAlign=baseline';
      end;
        //
      if Length(AText) = 0 then
      begin
        AText := Text;
      end;

      Result := TIWHTMLTag.CreateTag(iif(Header, 'TH', 'TD'));
      try
        Result.Add(LVAlign);
           // Always render alignment, default is different or TH and TD
        Result.AddStringParam('align', IWGridAlignments[Alignment]);
        Result.Add(iif(not Wrap, 'NOWRAP'));
        Result.Add(iif((Width <> '') and (Width <> '0'), 'Width=' + Width));
        Result.Add(iif((Height <> '') and (Height <> '0'), 'Height=' + Height));
        Result.Add(HTMLColor(BGColor, 'BGColor'));
        Result.Add(iif(Length(Font.CSSStyle) > 0, 'CLASS=' + Font.CSSStyle));


           // FontToString checks Font.Enabled
        if Font.CSSStyle = '' then
        begin
          if Result.Contents.AddTagAsObject(Font.FontToStringTag(AText)) = nil then
            Result.Contents.AddText(AText);
        end
        else
        begin
          Result.Contents.AddText(AText);
        end;
      except
        FreeAndNil(Result);
        raise;
      end;
    finally
      Grid := nil;
    end;
  end;
end;

procedure TIWCalendarCell.SetCellDate(const Value: TDateTime);
begin
  FCellDate := Value;
end;

procedure TIWCalendarCell.SetDateInformation(const Value: TStringList);
begin
  FDateInformation.Assign(Value);
end;

{ TIWCalendarCells }

function TIWCalendarCells.Add: TIWCalendarCell;
begin
  Result := TIWCalendarCell(inherited Add);
end;

function TIWCalendarCells.Owner: TPersistent;
begin
  Result := inherited GetOwner;
end;

end.


